/** @file   FileHasher.cpp
 * @brief   Implementation of FileHasher - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#include "FileHasher.h"
#include <fstream>
using std::string;
using std::ifstream;
using std::ofstream;
using std::ios;

namespace eng2d {

/** Constructs new file hasher
 */
FileHasher::FileHasher( const std::string& filename ) :
  m_filename( filename )
{
}



/** Copy constructor
 */
FileHasher::FileHasher( const FileHasher& hf ) :
  m_filename( hf.m_filename )
{
}



/** Destructor
 */
FileHasher::~FileHasher()
{
}



/** operator =
 */
FileHasher& FileHasher::operator = ( const FileHasher& hf )
{
  if ( this != &hf ) {
    m_filename = hf.m_filename;
  }
  return *this;
}



//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************

/** Inserts the hashcode after the "[END]" tag
 */
void FileHasher::set()
{
  HashNum n = hash();
  
  // Write the hashnumber at the end of the file
  ofstream fout( m_filename.c_str(), ios::app );
  if ( !fout ) {
    // Error
    return;
  }
  fout << n;
  fout.close();
}



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************

/** Checks the hashcode of the file 'm_filename' and return 'true'
 * if it's invalid
 */
bool FileHasher::check() const
{
#ifndef ENG2D_DISABLE_HASHCODES
  HashNum realHash = this->hash();
  
  // Read the hashcode that's saved in the file
  ifstream fin( m_filename.c_str() );
  if ( !fin ) {
    return false;
  }
  string tmp;
  while ( true ) {
    fin >> tmp;
    if ( tmp == "[END]") {
      // The next thing we read from the file, should be the hashnumber
      string hashstr;
      fin >> hashstr;
      if ( atol(hashstr.c_str()) != realHash ) {
        fin.close();
        return true;
      } else {
        fin.close();
        return false;
      }
    }
    if ( fin.eof() ) {
      break;
    }
  }
  fin.close();
  return true;
#else
  return false;
#endif
}



//********************************************************************
//                                                                   *
//      Private methods                                              *
//                                                                   *
//********************************************************************

/** Reads the file 'm_filename' and calculates the hashcode for it.
 */
FileHasher::HashNum FileHasher::hash() const
{
  ifstream fin( m_filename.c_str() );

  HashNum h = 0;
  string tmp;
  int key = 0;
  while ( fin.eof() == false ) {
    fin >> tmp;
    if ( tmp == "[END]" ) {
      break;
    }
    for (int i=0; i < tmp.length(); i++) {

      int asciiCode = static_cast<int>( tmp.at(i) );
      int r = asciiCode % 7;

      key ^= asciiCode;
      key ^= ( asciiCode << 8 );
      key ^= ( asciiCode << 16 );
      key = key << r;

      h += key;
      h = h & 0x3fffffff;
    }
  }
  return h;
}


} // end of namespace
